#include "stm32h7xx.h"
#include <math.h>
#include <string.h>
#if !defined  (HSE_VALUE)
#define HSE_VALUE    ((uint32_t)25000000) /*!< Value of the External oscillator in Hz */
#endif /* HSE_VALUE */

#if !defined  (CSI_VALUE)
  #define CSI_VALUE    ((uint32_t)4000000) /*!< Value of the Internal oscillator in Hz*/
#endif /* CSI_VALUE */

#if !defined  (HSI_VALUE)
  #define HSI_VALUE    ((uint32_t)64000000) /*!< Value of the Internal oscillator in Hz*/
#endif /* HSI_VALUE */


extern unsigned char itcm_text_start;
extern unsigned char itcm_text_end;
extern unsigned char itcm_data;

extern size_t __stack_top[];
extern size_t __ram_vectors_start[];
extern size_t __ram_vectors_end[];
extern size_t __reloc_ivt_start[];


#define VECT_TAB_SRAM
#define VECT_TAB_OFFSET  0x00000000UL

uint32_t SystemCoreClock = 480000000;
uint32_t SystemD2Clock = 480000000;
const uint8_t D1CorePrescTable[16] = { 0, 0, 0, 0, 1, 2, 3, 4, 1, 2, 3, 4, 6, 7, 8, 9 };


void SystemInit(void) {

	/* FPU settings ------------------------------------------------------------*/
    SCB->CPACR |= ((3UL << (10*2))|(3UL << (11*2)));  /* set CP10 and CP11 Full Access */

	RCC->CR |= RCC_CR_HSION;

	/* Reset CFGR register */
	RCC->CFGR = 0x00000000;

	/* Reset HSEON, CSSON , CSION,RC48ON, CSIKERON PLL1ON, PLL2ON and PLL3ON bits */
	RCC->CR &= 0xEAF6ED7FU;

	/* Reset D1CFGR register */
	RCC->D1CFGR = 0x00000000;

	/* Reset D2CFGR register */
	RCC->D2CFGR = 0x00000000;

	/* Reset D3CFGR register */
	RCC->D3CFGR = 0x00000000;

	/* Reset PLLCKSELR register */
	RCC->PLLCKSELR = 0x00000000;

	/* Reset PLLCFGR register */
	RCC->PLLCFGR = 0x00000000;
	/* Reset PLL1DIVR register */
	RCC->PLL1DIVR = 0x00000000;
	/* Reset PLL1FRACR register */
	RCC->PLL1FRACR = 0x00000000;

	/* Reset PLL2DIVR register */
	RCC->PLL2DIVR = 0x00000000;

	RCC->PLL2FRACR = 0x00000000;
	/* Reset PLL3DIVR register */
	RCC->PLL3DIVR = 0x00000000;

	/* Reset PLL3FRACR register */
	RCC->PLL3FRACR = 0x00000000;

	/* Reset HSEBYP bit */
	RCC->CR &= 0xFFFBFFFFU;

	/* Disable all interrupts */
	RCC->CIER = 0x00000000;

#if (STM32H7_DEV_ID == 0x450UL)
	/* dual core CM7 or single core line */
	if ((DBGMCU->IDCODE & 0xFFFF0000U) < 0x20000000U){
		/* if stm32h7 revY*/
		/* Change  the switch matrix read issuing capability to 1 for the AXI SRAM target (Target 7) */
		*((__IO uint32_t*) 0x51008108) = 0x000000001U;
	}
#endif

#if defined (DATA_IN_D2_SRAM)
#if defined(RCC_AHB2ENR_D2SRAM3EN)
  RCC->AHB2ENR |= (RCC_AHB2ENR_D2SRAM1EN | RCC_AHB2ENR_D2SRAM2EN | RCC_AHB2ENR_D2SRAM3EN);
#elif defined(RCC_AHB2ENR_D2SRAM2EN)
  RCC->AHB2ENR |= (RCC_AHB2ENR_D2SRAM1EN | RCC_AHB2ENR_D2SRAM2EN);
#else
  RCC->AHB2ENR |= (RCC_AHB2ENR_AHBSRAM1EN | RCC_AHB2ENR_AHBSRAM2EN);
#endif

  tmpreg = RCC->AHB2ENR;
  (void) tmpreg;
#endif

#if defined(DUAL_CORE) && defined(CORE_CM4)

#ifdef VECT_TAB_SRAM
  SCB->VTOR = D2_AHBSRAM_BASE | VECT_TAB_OFFSET;
#else
  SCB->VTOR = FLASH_BANK2_BASE | VECT_TAB_OFFSET;
#endif

#else

	/* Configure the Vector Table location add offset address for cortex-M7 ------------------*/
#ifdef VECT_TAB_SRAM
  size_t * p_src;
  size_t * p_dst;
  for (p_src = __reloc_ivt_start, p_dst = __ram_vectors_start; p_dst < __ram_vectors_end; )
  {
      *p_dst++ = *p_src++;
  }

  __disable_irq();
  SCB->VTOR = (uint32_t)__ram_vectors_start;
  __ISB();
  __DSB();
  __enable_irq();
  __DSB();
  __ISB();
#else
	SCB->VTOR = FLASH_BANK1_BASE | VECT_TAB_OFFSET; /* Vector Table Relocation in Internal FLASH */
#endif

#endif /*DUAL_CORE && CORE_CM4*/

}


void SystemCoreClockUpdate(void) {
	uint32_t pllp, pllsource, pllm, pllfracen, hsivalue, tmp;
	uint32_t common_system_clock;
	float_t fracn1, pllvco;

	/* Get SYSCLK source -------------------------------------------------------*/

	switch (RCC->CFGR & RCC_CFGR_SWS) {
	case RCC_CFGR_SWS_HSI: /* HSI used as system clock source */
		common_system_clock = (uint32_t) (HSI_VALUE
				>> ((RCC->CR & RCC_CR_HSIDIV) >> 3));
		break;

	case RCC_CFGR_SWS_CSI: /* CSI used as system clock  source */
		common_system_clock = CSI_VALUE;
		break;

	case RCC_CFGR_SWS_HSE: /* HSE used as system clock  source */
		common_system_clock = HSE_VALUE;
		break;

	case RCC_CFGR_SWS_PLL1: /* PLL1 used as system clock  source */

		/* PLL_VCO = (HSE_VALUE or HSI_VALUE or CSI_VALUE/ PLLM) * PLLN
		 SYSCLK = PLL_VCO / PLLR
		 */
		pllsource = (RCC->PLLCKSELR & RCC_PLLCKSELR_PLLSRC);
		pllm = ((RCC->PLLCKSELR & RCC_PLLCKSELR_DIVM1) >> 4);
		pllfracen = ((RCC->PLLCFGR & RCC_PLLCFGR_PLL1FRACEN)
				>> RCC_PLLCFGR_PLL1FRACEN_Pos);
		fracn1 = (float_t) (uint32_t) (pllfracen
				* ((RCC->PLL1FRACR & RCC_PLL1FRACR_FRACN1) >> 3));

		if (pllm != 0U) {
			switch (pllsource) {
			case RCC_PLLCKSELR_PLLSRC_HSI: /* HSI used as PLL clock source */

				hsivalue = (HSI_VALUE >> ((RCC->CR & RCC_CR_HSIDIV) >> 3));
				pllvco =
						((float_t) hsivalue / (float_t) pllm)
								* ((float_t) (uint32_t) (RCC->PLL1DIVR
										& RCC_PLL1DIVR_N1)
										+ (fracn1 / (float_t) 0x2000)
										+ (float_t) 1);

				break;

			case RCC_PLLCKSELR_PLLSRC_CSI: /* CSI used as PLL clock source */
				pllvco =
						((float_t) CSI_VALUE / (float_t) pllm)
								* ((float_t) (uint32_t) (RCC->PLL1DIVR
										& RCC_PLL1DIVR_N1)
										+ (fracn1 / (float_t) 0x2000)
										+ (float_t) 1);
				break;

			case RCC_PLLCKSELR_PLLSRC_HSE: /* HSE used as PLL clock source */
				pllvco =
						((float_t) HSE_VALUE / (float_t) pllm)
								* ((float_t) (uint32_t) (RCC->PLL1DIVR
										& RCC_PLL1DIVR_N1)
										+ (fracn1 / (float_t) 0x2000)
										+ (float_t) 1);
				break;

			default:
				pllvco =
						((float_t) CSI_VALUE / (float_t) pllm)
								* ((float_t) (uint32_t) (RCC->PLL1DIVR
										& RCC_PLL1DIVR_N1)
										+ (fracn1 / (float_t) 0x2000)
										+ (float_t) 1);
				break;
			}
			pllp = (((RCC->PLL1DIVR & RCC_PLL1DIVR_P1) >> 9) + 1U);
			common_system_clock =
					(uint32_t) (float_t) (pllvco / (float_t) pllp);
		} else {
			common_system_clock = 0U;
		}
		break;

	default:
		common_system_clock = CSI_VALUE;
		break;
	}

	/* Compute SystemClock frequency --------------------------------------------------*/
#if defined (RCC_D1CFGR_D1CPRE)
	tmp = D1CorePrescTable[(RCC->D1CFGR & RCC_D1CFGR_D1CPRE)
			>> RCC_D1CFGR_D1CPRE_Pos];

	/* common_system_clock frequency : CM7 CPU frequency  */
	common_system_clock >>= tmp;

	/* SystemD2Clock frequency : CM4 CPU, AXI and AHBs Clock frequency  */
	SystemD2Clock = (common_system_clock
			>> ((D1CorePrescTable[(RCC->D1CFGR & RCC_D1CFGR_HPRE)
					>> RCC_D1CFGR_HPRE_Pos]) & 0x1FU));

#else
  tmp = D1CorePrescTable[(RCC->CDCFGR1 & RCC_CDCFGR1_CDCPRE)>> RCC_CDCFGR1_CDCPRE_Pos];

  /* common_system_clock frequency : CM7 CPU frequency  */
  common_system_clock >>= tmp;

  /* SystemD2Clock frequency : AXI and AHBs Clock frequency  */
  SystemD2Clock = (common_system_clock >> ((D1CorePrescTable[(RCC->CDCFGR1 & RCC_CDCFGR1_HPRE)>> RCC_CDCFGR1_HPRE_Pos]) & 0x1FU));

#endif

#if defined(DUAL_CORE) && defined(CORE_CM4)
  SystemCoreClock = SystemD2Clock;
#else
	SystemCoreClock = common_system_clock;
#endif /* DUAL_CORE && CORE_CM4 */
}

/**
 * @}
 */

/**
 * @}
 */

/**
 * @}
 */
/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
