/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * mcurses.c - mcurses lib
 *
 * Copyright (c) 2011-2015 Frank Meyer - frank(at)fli4l.de
 *
 * Revision History:
 * V1.0 2015 xx xx Frank Meyer, original version
 * V1.1 2017 01 13 ChrisMicro, addepted as Arduino library, MCU specific functions removed
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mcurses.h"


#define SEQ_CSI                                 PSTR("\033[")                   // code introducer
#define SEQ_CLEAR                               PSTR("\033[2J")                 // clear screen
#define SEQ_CLRTOBOT                            PSTR("\033[J")                  // clear to bottom
#define SEQ_CLRTOEOL                            PSTR("\033[K")                  // clear to end of line
#define SEQ_DELCH                               PSTR("\033[P")                  // delete character
#define SEQ_NEXTLINE                            PSTR("\033E")                   // goto next line (scroll up at end of scrolling region)
#define SEQ_INSERTLINE                          PSTR("\033[L")                  // insert line
#define SEQ_DELETELINE                          PSTR("\033[M")                  // delete line
#define SEQ_ATTRSET                             PSTR("\033[0")                  // set attributes, e.g. "\033[0;7;1m"
#define SEQ_ATTRSET_REVERSE                     PSTR(";7")                      // reverse
#define SEQ_ATTRSET_UNDERLINE                   PSTR(";4")                      // underline
#define SEQ_ATTRSET_BLINK                       PSTR(";5")                      // blink
#define SEQ_ATTRSET_BOLD                        PSTR(";1")                      // bold
#define SEQ_ATTRSET_DIM                         PSTR(";2")                      // dim
#define SEQ_ATTRSET_FCOLOR                      PSTR(";3")                      // forground color
#define SEQ_ATTRSET_BCOLOR                      PSTR(";4")                      // background color

#define SEQ_ATTRSET_HIGH_FCOLOR                 PSTR(";9")                      // forground color
#define SEQ_ATTRSET_HIGH_BCOLOR                 PSTR(";10")                     // background color

#define SEQ_INSERT_MODE                         PSTR("\033[4h")                 // set insert mode
#define SEQ_REPLACE_MODE                        PSTR("\033[4l")                 // set replace mode
#define SEQ_RESET_SCRREG                        PSTR("\033[r")                  // reset scrolling region
#define SEQ_LOAD_G1                             PSTR("\033)0")                  // load G1 character set
#define SEQ_CURSOR_VIS                          PSTR("\033[?25")                // set cursor visible/not visible

static uint_fast8_t                             mcurses_scrl_start = 0;         // start of scrolling region, default is 0
static uint_fast8_t                             mcurses_scrl_end = LINES - 1;   // end of scrolling region, default is last line
static uint_fast8_t                             mcurses_nodelay;                // nodelay flag
static uint_fast8_t                             mcurses_halfdelay;              // halfdelay value, in tenths of a second

uint_fast8_t                                    mcurses_is_up = 0;              // flag: mcurses is up
uint_fast8_t                                    mcurses_cury = 0xff;            // current y position of cursor, public (getyx())
uint_fast8_t                                    mcurses_curx = 0xff;            // current x position of cursor, public (getyx())

static void                                     mcurses_puts_P (const char *);
static void                                     mcurses_puts_DISP (const char *);


static uint_fast8_t mcurses_phyio_init (void)
{
  return 0;
}


static void mcurses_phyio_done (void)
{
	
}

static void mcurses_phyio_putc (uint_fast8_t ch)
{
	pushtxDispData(&txDispData, ch, yGrid[NumberWindows], xGrid[NumberWindows]);
}


static void mcurses_phyio_nodelay (uint_fast8_t flag)
{
    mcurses_nodelay = flag;
}
/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * PHYIO: set/reset halfdelay (AVR)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void mcurses_phyio_halfdelay (uint_fast8_t tenths)
{
    mcurses_halfdelay = tenths;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * PHYIO: flush output (AVR)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void mcurses_phyio_flush_output ()
{
	
}


static void
mcurses_putc (uint_fast8_t ch)
{
    mcurses_phyio_putc (ch);
}


static void
mcurses_putcDISP (uint_fast8_t ch)
{
	pushDispDataUpdate(&DispDataUpdate,ch);
}

static void
mcurses_putcMenu (uint_fast8_t ch)
{
	//while(sizeStackMenu(&Menu) > 5000);
	pushMenu(&Menu, ch, MenuY, MenuX);
}


static void
mcurses_putcMenuUpdate (uint_fast8_t ch)
{
	pushMenuUpdate(&MenuUpdate, ch);
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * INTERN: put a string from flash (raw)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
mcurses_puts_P (const char * str)
{
    uint_fast8_t ch;

    while ((ch = pgm_read_byte(str)) != '\0')
    {
        mcurses_putc (ch);
        str++;
    }
}


static void
mcurses_puts_DISP (const char * str)
{
    uint_fast8_t ch;

    while ((ch = pgm_read_byte(str)) != '\0')
    {
    	pushDispDataUpdate(&DispDataUpdate,ch);
        str++;
    }
}


static void
mcurses_puts_Menu (const char * str)
{
    uint_fast8_t ch;

    while ((ch = pgm_read_byte(str)) != '\0')
    {
    	//while(sizeStackMenu(&Menu) > 5000);
    	pushMenu(&Menu, ch, MenuY, MenuX);
        str++;
    }
}

static void
mcurses_puts_MenuUpdate (const char * str)
{
    uint_fast8_t ch;

    while ((ch = pgm_read_byte(str)) != '\0')
    {
    	pushMenuUpdate(&MenuUpdate, ch);
        str++;
    }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * INTERN: put a 3/2/1 digit integer number (raw)
 *
 * Here we don't want to use sprintf (too big on AVR/Z80) or itoa (not available on Z80)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
mcurses_puti (uint_fast8_t i)
{
    uint_fast8_t ii;

    if (i >= 10)
    {
        if (i >= 100)
        {
            ii = i / 100;
            mcurses_putc (ii + '0');
            i -= 100 * ii;
        }

        ii = i / 10;
        mcurses_putc (ii + '0');
        i -= 10 * ii;
    }

    mcurses_putc (i + '0');
}


static void
mcurses_putiDISP (uint_fast8_t i)
{
    uint_fast8_t ii;

    if (i >= 10)
    {
        if (i >= 100)
        {
            ii = i / 100;
            pushDispDataUpdate(&DispDataUpdate,(ii + '0'));
            i -= 100 * ii;
        }

        ii = i / 10;
        pushDispDataUpdate(&DispDataUpdate,(ii + '0'));
        i -= 10 * ii;
    }

    pushDispDataUpdate(&DispDataUpdate,(i + '0'));
}


static void
mcurses_putiMenu (uint_fast8_t i)
{
    uint_fast8_t ii;

    if (i >= 10)
    {
        if (i >= 100)
        {
            ii = i / 100;
            pushMenu(&Menu,(ii + '0'), MenuY, MenuX);

            i -= 100 * ii;
        }

        ii = i / 10;
        pushMenu(&Menu,(ii + '0'),MenuY, MenuX);
        i -= 10 * ii;
    }
    pushMenu(&Menu,(i + '0'), MenuY, MenuX);
}


static void
mcurses_putiMenuUpdate (uint_fast8_t i)
{
    uint_fast8_t ii;

    if (i >= 10)
    {
        if (i >= 100)
        {
            ii = i / 100;
            pushMenuUpdate(&MenuUpdate,(ii + '0'));

            i -= 100 * ii;
        }

        ii = i / 10;
        pushMenuUpdate(&MenuUpdate,(ii + '0'));
        i -= 10 * ii;
    }

    pushMenuUpdate(&MenuUpdate,(i + '0'));
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * INTERN: addch or insch a character
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */


static void
mcurses_addch_or_insch (uint_fast8_t ch)
{
    mcurses_putc (ch);
}


static void
mcurses_addch_or_inschDISP (uint_fast8_t ch)
{
    mcurses_putcDISP (ch);
}


static void
mcurses_addch_or_inschMenu (uint_fast8_t ch)
{
    mcurses_putcMenu(ch);
}


static void
mcurses_addch_or_inschMenuUpdate (uint_fast8_t ch)
{
    mcurses_putcMenuUpdate (ch);
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * INTERN: set scrolling region (raw)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
mysetscrreg (uint_fast8_t top, uint_fast8_t bottom)
{
    if (top == bottom)
    {
        mcurses_puts_P (SEQ_RESET_SCRREG);                                      // reset scrolling region
    }
    else
    {
        mcurses_puts_P (SEQ_CSI);
        mcurses_puti (top + 1);
        mcurses_putc (';');
        mcurses_puti (bottom + 1);
        mcurses_putc ('r');
    }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * move cursor (raw)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
mymove (uint_fast8_t y, uint_fast8_t x)
{
    mcurses_puts_P (SEQ_CSI);
    mcurses_puti (y+1);
    mcurses_putc (';');
    mcurses_puti (x+1);
    mcurses_putc ('H');
}


static void
mymoveDISP (uint_fast8_t y, uint_fast8_t x)
{
    mcurses_puts_DISP (SEQ_CSI);
    mcurses_putiDISP (y+1);
    mcurses_putcDISP (';');
    mcurses_putiDISP (x+1);
    mcurses_putcDISP ('H');
}


static void
mymoveMenu (uint_fast8_t y, uint_fast8_t x)
{
    mcurses_puts_Menu (SEQ_CSI);
    mcurses_putiMenu (y+1);
    mcurses_putcMenu (';');
    mcurses_putiMenu (x+1);
    mcurses_putcMenu ('H');
}


static void
mymoveMenuUpdate (uint_fast8_t y, uint_fast8_t x)
{
    mcurses_puts_MenuUpdate (SEQ_CSI);
    mcurses_putiMenuUpdate (y+1);
    mcurses_putcMenuUpdate (';');
    mcurses_putiMenuUpdate (x+1);
    mcurses_putcMenuUpdate ('H');
}
/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: initialize
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
uint_fast8_t
initscr (void)
{
    uint_fast8_t rtc;

    if (mcurses_phyio_init ())
    {
        mcurses_puts_P (SEQ_LOAD_G1);                                               // load graphic charset into G1
        attrset (A_NORMAL);
        clear ();
        move (0, 0);
        mcurses_is_up = 1;
        rtc = OK;
    }
    else
    {
        rtc = ERR;
    }
    return rtc;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: add character
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
addch (uint_fast8_t ch)
{
    mcurses_addch_or_insch (ch);
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: add string
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
addstr (const char * str)
{
    while (*str)
    {
        mcurses_addch_or_insch (*str++);
    }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: add string
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
addstr_P (const char * str)
{
    uint_fast8_t ch;

    while ((ch = pgm_read_byte(str)) != '\0')
    {
        mcurses_addch_or_insch (ch);
        str++;
    }
}


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: set attribute(s)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
attrset (uint_fast16_t attr)
{
        uint_fast8_t        idx;

        mcurses_puts_P (SEQ_ATTRSET);

        idx = (attr & F_COLOR) >> 8;

        if (idx >= 0 && idx <= 7)
        {
            mcurses_puts_P (SEQ_ATTRSET_FCOLOR);
            mcurses_putc (idx + '0');
        }

        if (idx > 7 && idx <= 15)
        {
            mcurses_puts_P (SEQ_ATTRSET_HIGH_FCOLOR);
            mcurses_putc (idx - 8 + '0');
        }

        idx = (attr & B_COLOR) >> 12;

        if (idx >= 0 && idx <= 7)
        {
            mcurses_puts_P (SEQ_ATTRSET_BCOLOR);
            mcurses_putc (idx + '0');
        }

        if (idx > 7 && idx <= 15)
        {
            mcurses_puts_P (SEQ_ATTRSET_HIGH_BCOLOR);
            mcurses_putc (idx - 8 + '0');
        }

        if (attr & A_REVERSE)
        {
            mcurses_puts_P (SEQ_ATTRSET_REVERSE);
        }
        if (attr & A_UNDERLINE)
        {
            mcurses_puts_P (SEQ_ATTRSET_UNDERLINE);
        }
        if (attr & A_BLINK)
        {
            mcurses_puts_P (SEQ_ATTRSET_BLINK);
        }
        if (attr & A_BOLD)
        {
            mcurses_puts_P (SEQ_ATTRSET_BOLD);
        }
        if (attr & A_DIM)
        {
            mcurses_puts_P (SEQ_ATTRSET_DIM);
        }
        mcurses_putc ('m');
}


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: move cursor
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
move (uint_fast8_t y, uint_fast8_t x)
{
    mymove (y, x);
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: delete line
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
deleteln (void)
{
    mysetscrreg (mcurses_scrl_start, mcurses_scrl_end);                         // set scrolling region
    mymove (mcurses_cury, 0);                                                   // goto to current line
    mcurses_puts_P (SEQ_DELETELINE);                                            // delete line
    mysetscrreg (0, 0);                                                         // reset scrolling region
    mymove (mcurses_cury, mcurses_curx);                                        // restore position
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: insert line
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
insertln (void)
{
    mysetscrreg (mcurses_cury, mcurses_scrl_end);                               // set scrolling region
    mymove (mcurses_cury, 0);                                                   // goto to current line
    mcurses_puts_P (SEQ_INSERTLINE);                                            // insert line
    mysetscrreg (0, 0);                                                         // reset scrolling region
    mymove (mcurses_cury, mcurses_curx);                                        // restore position
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: scroll
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
scroll (void)
{
    mysetscrreg (mcurses_scrl_start, mcurses_scrl_end);                         // set scrolling region
    mymove (mcurses_scrl_end, 0);                                               // goto to last line of scrolling region
    mcurses_puts_P (SEQ_NEXTLINE);                                              // next line
    mysetscrreg (0, 0);                                                         // reset scrolling region
    mymove (mcurses_cury, mcurses_curx);                                        // restore position
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: clear
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
clear (void)
{
    mcurses_puts_P (SEQ_CLEAR);
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: clear to bottom of screen
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
clrtobot (void)
{
    mcurses_puts_P (SEQ_CLRTOBOT);
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: clear to end of line
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
clrtoeol (void)
{
    mcurses_puts_P (SEQ_CLRTOEOL);
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: delete character at cursor position
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
delch (void)
{
    mcurses_puts_P (SEQ_DELCH);
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: insert character
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
insch (uint_fast8_t ch)
{
    mcurses_addch_or_insch (ch);
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: set scrolling region
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
setscrreg (uint_fast8_t t, uint_fast8_t b)
{
    mcurses_scrl_start = t;
    mcurses_scrl_end = b;
}

void
curs_set (uint_fast8_t visibility)
{
    mcurses_puts_P (SEQ_CURSOR_VIS);

    if (visibility == 0)
    {
        mcurses_putc ('l');
    }
    else
    {
        mcurses_putc ('h');
    }
}


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: refresh: flush output
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
refresh (void)
{
    mcurses_phyio_flush_output ();
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: set/reset nodelay
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
nodelay (uint_fast8_t flag)
{
    if (mcurses_nodelay != flag)
    {
        mcurses_phyio_nodelay (flag);
    }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: set/reset halfdelay
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
halfdelay (uint_fast8_t tenths)
{
    mcurses_phyio_halfdelay (tenths);
}


void
endwin (void)
{
    move (LINES - 1, 0);                                                        // move cursor to last line
    clrtoeol ();                                                                // clear this line
    mcurses_putc ('\017');                                                      // switch to G0 set
    curs_set (TRUE);                                                            // show cursor
    mcurses_puts_P(SEQ_REPLACE_MODE);                                           // reset insert mode
    refresh ();                                                                 // flush output
    mcurses_phyio_done ();                                                      // end of physical I/O
    mcurses_is_up = 0;
}


/*  ன ⮪*/
void
clearDISP(void)
{
	mcurses_puts_DISP (SEQ_CLEAR);
}


void
moveDISP (uint_fast8_t y, uint_fast8_t x)
{
        mymoveDISP(y, x);
}

void
curs_setDISP(uint_fast8_t visibility)
{
    mcurses_puts_DISP (SEQ_CURSOR_VIS);

    if (visibility == 0)
    {
        mcurses_putcDISP ('l');
    }
    else
    {
        mcurses_putcDISP ('h');
    }
}

void
attrsetDISP(uint_fast16_t attr)
{
    uint_fast8_t        idx;

        mcurses_puts_DISP (SEQ_ATTRSET);

        idx = (attr & F_COLOR) >> 8;

        if (idx >= 0 && idx <= 7)
        {
            mcurses_puts_DISP (SEQ_ATTRSET_FCOLOR);
            mcurses_putcDISP (idx + '0');
        }

        if (idx > 7 && idx <= 15)
        {
            mcurses_puts_DISP (SEQ_ATTRSET_HIGH_FCOLOR);
            mcurses_putcDISP (idx - 8 + '0');
        }

        idx = (attr & B_COLOR) >> 12;

        if (idx >= 0 && idx <= 7)
        {
            mcurses_puts_DISP (SEQ_ATTRSET_BCOLOR);
            mcurses_putcDISP (idx + '0');
        }

        if (idx > 7 && idx <= 15)
        {
            mcurses_puts_DISP (SEQ_ATTRSET_HIGH_BCOLOR);
            mcurses_putcDISP (idx - 8 + '0');
        }

        if (attr & A_REVERSE)
        {
            mcurses_puts_DISP (SEQ_ATTRSET_REVERSE);
        }
        if (attr & A_UNDERLINE)
        {
            mcurses_puts_DISP (SEQ_ATTRSET_UNDERLINE);
        }
        if (attr & A_BLINK)
        {
            mcurses_puts_DISP (SEQ_ATTRSET_BLINK);
        }
        if (attr & A_BOLD)
        {
            mcurses_puts_DISP (SEQ_ATTRSET_BOLD);
        }
        if (attr & A_DIM)
        {
            mcurses_puts_DISP (SEQ_ATTRSET_DIM);
        }
        mcurses_putcDISP ('m');
}


void
addchDISP(uint_fast8_t ch)
{
	mcurses_addch_or_inschDISP (ch);
}



void
clearMenu(void)
{
	mcurses_puts_Menu (SEQ_CLEAR);
}

void
moveMenu(uint_fast8_t y, uint_fast8_t x)
{
    mymoveMenu(y, x);
}

void
curs_setMenu(uint_fast8_t visibility)
{
    mcurses_puts_Menu (SEQ_CURSOR_VIS);

    if (visibility == 0)
    {
        mcurses_putcMenu ('l');
        onOffCursor = 0;
    }
    else
    {
        mcurses_putcMenu ('h');
        onOffCursor = 1;
    }

}

void
attrsetMenu(uint_fast16_t attr)
{
    uint_fast8_t        idx;

        mcurses_puts_Menu (SEQ_ATTRSET);

        idx = (attr & F_COLOR) >> 8;

        if (idx >= 0 && idx <= 7)
        {
            mcurses_puts_Menu (SEQ_ATTRSET_FCOLOR);
            mcurses_putcMenu (idx + '0');
        }

        idx = (attr & B_COLOR) >> 12;

        if (idx >= 0 && idx <= 7)
        {
            mcurses_puts_Menu (SEQ_ATTRSET_BCOLOR);
            mcurses_putcMenu (idx + '0');
        }

        if (attr & A_REVERSE)
        {
            mcurses_puts_Menu (SEQ_ATTRSET_REVERSE);
        }
        if (attr & A_UNDERLINE)
        {
            mcurses_puts_Menu (SEQ_ATTRSET_UNDERLINE);
        }
        if (attr & A_BLINK)
        {
            mcurses_puts_Menu (SEQ_ATTRSET_BLINK);
        }
        if (attr & A_BOLD)
        {
            mcurses_puts_Menu (SEQ_ATTRSET_BOLD);
        }
        if (attr & A_DIM)
        {
            mcurses_puts_Menu (SEQ_ATTRSET_DIM);
        }
        mcurses_putcMenu ('m');
}

void
addchMenu(uint_fast8_t ch)
{
	mcurses_addch_or_inschMenu(ch);
}

void
addstrMenu (const char *str)
{
    while (*str)
    {
        mcurses_addch_or_inschMenu(*str++);
    }
}



void
clearMenuUpdate(void)
{
	mcurses_puts_MenuUpdate(SEQ_CLEAR);
}


void
moveMenuUpdate(uint_fast8_t y, uint_fast8_t x)
{
    mymoveMenuUpdate (y, x);
}


void
curs_setMenuUpdate(uint_fast8_t visibility)
{
    mcurses_puts_MenuUpdate (SEQ_CURSOR_VIS);

    if (visibility == 0)
    {
        mcurses_putcMenuUpdate ('l');
    }
    else
    {
        mcurses_putcMenuUpdate ('h');
    }
}


void
attrsetMenuUpdate(uint_fast16_t attr)
{
    uint_fast8_t        idx;

        mcurses_puts_MenuUpdate (SEQ_ATTRSET);

        idx = (attr & F_COLOR) >> 8;

        if (idx >= 0 && idx <= 7)
        {
            mcurses_puts_MenuUpdate (SEQ_ATTRSET_FCOLOR);
            mcurses_putcMenuUpdate (idx + '0');
        }

        idx = (attr & B_COLOR) >> 12;

        if (idx >= 0 && idx <= 7)
        {
            mcurses_puts_MenuUpdate (SEQ_ATTRSET_BCOLOR);
            mcurses_putcMenuUpdate (idx + '0');
        }

        if (attr & A_REVERSE)
        {
            mcurses_puts_MenuUpdate (SEQ_ATTRSET_REVERSE);
        }
        if (attr & A_UNDERLINE)
        {
            mcurses_puts_MenuUpdate (SEQ_ATTRSET_UNDERLINE);
        }
        if (attr & A_BLINK)
        {
            mcurses_puts_MenuUpdate (SEQ_ATTRSET_BLINK);
        }
        if (attr & A_BOLD)
        {
            mcurses_puts_MenuUpdate (SEQ_ATTRSET_BOLD);
        }
        if (attr & A_DIM)
        {
            mcurses_puts_MenuUpdate (SEQ_ATTRSET_DIM);
        }
        mcurses_putcMenuUpdate ('m');
}


void
addchMenuUpdate(uint_fast8_t ch)
{
	mcurses_addch_or_inschMenuUpdate (ch);
}
